<?php
/**
 * scan_fix_bom_whitespace.php
 * ----------------------------------------------
 * Purpose: Detect and optionally fix UTF-8 BOM and leading whitespace
 * before the opening <?php tag in PHP files. These cause
 * "headers already sent" and break session_start()/header().
 *
 * Usage:
 *   1) Upload this file to public_html/technosky/
 *   2) Visit: https://yourdomain/technosky/scan_fix_bom_whitespace.php
 *      - Dry run: lists problematic files only.
 *   3) To fix: https://yourdomain/technosky/scan_fix_bom_whitespace.php?fix=1
 *      - Creates a backup .bak first and then rewrites the file without BOM
 *        and trims whitespace before the first "<?php".
 *
 * WARNING: Always take a backup before mass changes.
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

$base = __DIR__;                     // scan from current directory
$extensions = ['php'];               // only PHP files
$fix = isset($_GET['fix']);

function has_bom($bytes) {
    return substr($bytes, 0, 3) === "\xEF\xBB\xBF";
}

function find_php_open_tag_pos($bytes) {
    return strpos($bytes, "<?php");
}

function is_whitespace($bytes) {
    // treat spaces, tabs, CR, LF as whitespace
    return preg_match('/^[\x20\x09\x0D\x0A]*$/', $bytes) === 1;
}

function process_file($path, $fix) {
    $bytes = file_get_contents($path);
    $changed = false;

    $report = [
        'path' => $path,
        'bom'  => has_bom($bytes),
        'leading_ws' => false,
        'action' => 'none',
    ];

    // Check leading whitespace before first <?php
    $pos = find_php_open_tag_pos($bytes);
    if ($pos !== false && $pos > 0) {
        $before = substr($bytes, 0, $pos);
        if (is_whitespace($before)) {
            $report['leading_ws'] = true;
        }
    }

    if ($fix && ($report['bom'] || $report['leading_ws'])) {
        // make backup
        @copy($path, $path . '.bak');

        // strip BOM
        if ($report['bom']) {
            $bytes = substr($bytes, 3);
            $changed = true;
        }

        // trim whitespace BEFORE first <?php (only if whitespace)
        $pos = find_php_open_tag_pos($bytes);
        if ($pos !== false && $pos > 0) {
            $before = substr($bytes, 0, $pos);
            if (is_whitespace($before)) {
                $bytes = substr($bytes, $pos);
                $changed = true;
            }
        }

        if ($changed) {
            file_put_contents($path, $bytes);
            $report['action'] = 'fixed';
        }
    }

    return $report;
}

$rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($base, FilesystemIterator::SKIP_DOTS));
$reports = [];
foreach ($rii as $file) {
    if ($file->isDir()) continue;
    $path = $file->getPathname();
    $ext = strtolower(pathinfo($path, PATHINFO_EXTENSION));
    if (!in_array($ext, $extensions)) continue;

    $reports[] = process_file($path, $fix);
}

header('Content-Type: text/plain; charset=utf-8');

$mode = $fix ? 'FIX MODE' : 'DRY RUN';
echo "scan_fix_bom_whitespace.php - $mode\n";
echo str_repeat('=', 60) . "\n\n";

$issues = 0; $fixed = 0;
foreach ($reports as $r) {
    if ($r['bom'] || $r['leading_ws']) {
        $issues++;
        echo "[ISSUE] " . $r['path'] . "\n";
        echo "        BOM: " . ($r['bom'] ? 'YES' : 'no') . ", leading_ws: " . ($r['leading_ws'] ? 'YES' : 'no') . "\n";
        if ($r['action'] === 'fixed') { $fixed++; echo "        ACTION: fixed\n"; }
    }
}

if ($issues === 0) {
    echo "No BOM or leading whitespace issues found.\n";
} else {
    echo "\nSummary: $issues file(s) with issues" . ($fix ? ", $fixed fixed." : '. Use ?fix=1 to auto-fix.') . "\n";
}

?>
